import { Body, Controller, Delete, Get, Inject, Param, Post } from '@nestjs/common';
import { ApiBody, ApiOkResponse, ApiParam, ApiTags } from '@nestjs/swagger';
import { PrimeLogger, User } from 'src/framework';
import { AllowRoles, RoleShortNameEnum } from 'src/framework/infrastructure/decorators/allow-profiles/allow-roles.decorator';
import { ApiDefaultResponses } from 'src/framework/infrastructure/decorators/api-default-responses/api-default-responses.decorator';
import { PrimeUser } from 'src/framework/infrastructure/decorators/prime-user/prime-user.decorator';
import { KeywordAgileTenderService } from 'src/licitaapp/application/service/keyword-agile-tender-service/keyword-agile-tender-service.interface';
import { AgileKeyword, InsertAgileKeyword } from 'src/licitaapp/domain/entities';

@Controller('keyword-agile-tender')
@ApiTags('keyword-agile-tender')
export class KeywordAgileTenderController {

    private readonly LOGGER = new PrimeLogger(KeywordAgileTenderController.name);
    constructor(
        @Inject('KeywordAgileTenderService') private readonly keywordAgileTenderService: KeywordAgileTenderService,
    ) {}

    @Get('search-word/:word')
    @ApiDefaultResponses()
    @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
    @ApiOkResponse({ type: [String] })
    @ApiParam({ name: 'word', type: 'string' })
    async findByWord(
        @Param('word') word: string,
    ): Promise<String[]> {
        this.LOGGER.log(`Finding agile tender by word: ${word}`);
        return await this.keywordAgileTenderService.getAgileTenderByWord(word);
    }

    @Get('find-admin-values')
    @ApiOkResponse({ type: [String] })
    @ApiDefaultResponses()
    @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
    async findAdminValues(): Promise<string[]> {
        this.LOGGER.log(`findAdminValues`);
        return await this.keywordAgileTenderService.findAdminValues();
    }

    @Get('find-by-agile-tender/:agileTenderId')
    @ApiOkResponse({ type: [AgileKeyword] })
    @ApiDefaultResponses()
    @ApiParam({ name: 'agileTenderId', type: 'number' })
    @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
    async findByAgileTenderId(
        @Param('agileTenderId') agileTenderId: number,
    ): Promise<AgileKeyword[]> {
        return this.keywordAgileTenderService.findByAgileTenderId(agileTenderId);
    }


    @Delete('logical-remove/:keywordAgileTenderId')
    @ApiDefaultResponses()
    @ApiOkResponse({ type: String })
    @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
    @ApiParam({ name: 'keywordAgileTenderId', type: 'number' })
    async logicalRemove(
    @Param('keywordAgileTenderId') keywordAgileTenderId: number,
    ): Promise<String> {
    this.LOGGER.log(`logicalRemove keyword: ${keywordAgileTenderId}`);
    await this.keywordAgileTenderService.logicalRemove(keywordAgileTenderId);
    return 'true';
    }

    @Post()
    @ApiDefaultResponses()
    @AllowRoles(RoleShortNameEnum.USER, RoleShortNameEnum.ADMIN)
    @ApiBody({ type: InsertAgileKeyword })
    @ApiOkResponse({ type: String })
    async createKeyword(
        @PrimeUser() user: User,
        @Body() data: InsertAgileKeyword,
    ) {
      this.LOGGER.warn(`createKeyword keyword: ${JSON.stringify(data)} by admin user: ${user.id}`);
      await this.keywordAgileTenderService.save(data, user.id);
      return 'true';
  } 

}
